/**
 *@file ISR.c
 *
 *@brief Fichier contenant les interruptions de l'UART0/DMA ainsi que du bus CAN ainsi que la fonction permettant de les initialiser.
 *@details
 * Vous trouverez ici les interruptions EX_INTERRUPT_HANDLER(Uart0_ISR), EX_INTERRUPT_HANDLER(Uart0_tx_DMA_ISR), EX_INTERRUPT_HANDLER(Can_rx_ISR) ainsi que la fonction d'initialisation des interruptions : Init_Interrupts().
 *@warning Pour activer les interruptions, il faut  la fois avoir lanc les fonctions Init_Interrupts(), UART0_init() et init_CAN().
 *@author Bayoumi Chamaa et Comte-Gaz Quentin
 *@date Novembre 2013
 *@version 1.0
 */
 //--------------------------------------------------------------------------//
// Header files																//
//--------------------------------------------------------------------------//
#include <cdefBF534.h>
#include <sys/exception.h>
#include "uart.h"
#include "CAN.h"

//--------------------------------------------------------------------------//
// Variables globales														//
//--------------------------------------------------------------------------//

extern unsigned char buffer[100]; /* Buffer propre  l'uart (se remplissant en temps rel) */
extern unsigned char buffer_can[100]; /* Buffer dit par l'uart uniquement quand la trame est valide et utilis par le CAN */
extern int buffer_can_ready; // 0->Interdiction au CAN_IRQ de toucher au buffer | 1->CAN_IRQ peut toucher au buffer 
//En thorie, inutile car on a jou sur la priorit des interruptions pour garantir que le buffer sera valide lors de l'interruption CAN


int position_buffer=0; /*!< @brief Correspond  la position dans buffer[100] (ie : au nombre d'lements+1 dans le buffer[100]) */
int flag_irq_tx=0;     /*!< @brief Flag pour l'envoie d'une trame via UART0 par le biais du DMA9 */
int flag_fin_can=1;    /*!< @brief Si CAN est fini d'tre utilis =>1, sinon 0<br>Ce flag permet de ne pas envoyer "en mme temps" 2 mmes trames via bus CAN*/

//--------------------------------------------------------------------------//
// UART0 RX IRQ																//
//--------------------------------------------------------------------------//
/**
 *@fn EX_INTERRUPT_HANDLER(Uart0_ISR)
 *@brief Interruption de reception d'un char via l'UART 0 (GPS)
 *@details
 * Cette interruption rcupre les char via l'UART 0, les enregistrent dans buffer[100]
 * Lorsqu'un caractre de fin de trame est dtect, on verifie que la trame soit bien celle que l'on souhaite, on l'enregistre dans buffer_can[100] puis on active l'envoie via UART 0.
 * Le fait d'enregistrer la trame dans buffer_can[100] permet d'envoyer celle ci via CAN lorsqu'une demande CAN sera lance.
 */
EX_INTERRUPT_HANDLER(Uart0_ISR){
	int k=0;
	int uart0_flag=*pUART0_IIR;
	int mbID;
		
	uart0_flag&=0x06; //On ne garde que les bits de test

	if(uart0_flag==0x04){ //Si "Receive data ready" :
	
	   if(position_buffer>=100) //au cas o
	   	position_buffer=0; //Correspondra au nombre d'octets-1 de notre trame 
	   	
	   if(flag_irq_tx==0){ //Si on ne transmet pas :	
	   	buffer[position_buffer]=uart0_read();
	   	if(buffer[position_buffer]==0x0A){ //Si on est  la fin de la trame alors on reinitialise le buffer.
		 if(buffer[0]=='$' && buffer[1]=='G' 
	   		&& buffer[2]=='P' && buffer[3]=='G' 
	   		&& buffer[4]=='G' && buffer[5]=='A'){
	   		// Si notre trame est une trame  envoyer, alors on l'envoie
	   	   	flag_irq_tx=1;
	   	   	
	   		uart0_send(buffer[0]); //Envoie du premier lement via uart (qui va activer l'interruption uart tx)
	   		// ATTENTION, pour initialiser le cycle dma, il faut faire un uart_thr, d'o le uart0_send
	   	   	*pDMA9_X_COUNT=position_buffer+1; 
	   		*pDMA9_CONFIG|=0x01;
	    
	   		buffer_can_ready=0;// Interdire l'utilisation du buffer CAN pour l'interruption CAN
	   		for(k=0;k<position_buffer;k++)// Editer le buffer CAN (pour y mettre une trame valide)
	   			buffer_can[k]=buffer[k];
	   		buffer_can_ready=1;// Autoriser l'utilisation du buffer CAN pour l'interruption CAN		   	    
	   	}
	    position_buffer=0;
	   }
	   else
	   	position_buffer++;
	}
  }
}

//--------------------------------------------------------------------------//
// UART0 TX IRQ																//
//--------------------------------------------------------------------------//
/**
 *@fn EX_INTERRUPT_HANDLER(Uart0_tx_DMA_ISR)
 *@brief Interruption de transmission d'une trame via l'UART0-DMA9 (Hyperterminal du PC)
 *@details
 * Cette interruption se produit lorsque l'on a envoy le premier char de la trame via l'UART0.
 * Il reste donc  envoyer les autres lments grce au DMA (ce que l'on fait ici).
 */
EX_INTERRUPT_HANDLER(Uart0_tx_DMA_ISR){
    //Confirmer l'interruption
	*pDMA9_IRQ_STATUS = 0x01;
	flag_irq_tx=0;
}

//--------------------------------------------------------------------------//
// CAN RX IRQ																//
//--------------------------------------------------------------------------//
/**
 *@fn EX_INTERRUPT_HANDLER(Can_rx_ISR)
 *@brief Interruption de reception d'une demande CAN via le bus CAN
 *@details
 * Cette interruption se lance uniquement quand un des utilisateurs du bus CAN demande l'heure (id 20) ou la position GPS (id 21).
 * Si tel est le cas, on lui rpond en lui envoyant l'heure (id 20) ou la position GPS(id 21).
 * L'heure UTC est transmise sous forme ASCII : ab heure cd min ef sec -> abcdef(ASCII).
 * La position GPS a quand  elle sous forme binaire : abcd'ef.gh" N/S ijk'lm.nop" E/W -> donnes transmises via CAN 0xabcdefijklm0+K (K ayant pour LSB 0 si Nord ou 1 si Sud et pour 2eme LSB 0 si Ouest ou 1 si Est). 
 */
EX_INTERRUPT_HANDLER(Can_rx_ISR){
	unsigned char buffer_long_lat[8];
    int can_rx_flag=*pCAN_MBRIF1; // Flag mailbox 0->15
    int i=0;
	int nbr_mb_used=0;
	int nb_virgules=0;
	int position_virgules[6]={0,0,0,0,0,0};
	int is_point_in_hour=0;
	int nb_points=0;
	int position_points[3]={0,0,0};
	//can_rx_flag&=0x0003; // flag mailbox 0 et 1 
    *pCAN_MBRIF1 =0x0003; 
    
    // Configuration des mailbox du CAN
	//envoyer(trame via CAN) -->editer mailbox
	if(buffer_can_ready){   	
		*pCAN_TRS2 = 0x0000;
   		for(i=0; i<100; i++){
   			if(buffer_can[i]==','){
   				position_virgules[nb_virgules]=i;
   				nb_virgules++;
   				if(nb_virgules>=6) //Sortir de la boucle si on a le nombre de virgules qu'il nous faut.
   					i=100;
   			}
   			if(buffer_can[i]=='.' && nb_points<=3){
   				position_points[nb_points]=i;
   				nb_points++;
   			}		
   		}
   			
    if((can_rx_flag&0x0001)==0x0001){ //Demande de l'heure
    	//Envoyer l'heure via la l'id 20 si la trame est valide :
    	if((nb_virgules>=2) && ((position_virgules[1]-position_virgules[0])>4)){ //si la trame est valide : 
				//ENVOIE DE L'HEURE EN ASCII AVEC UNE MAILBOX
				preparation_mailbox(buffer_can+position_virgules[0]+1,6,20);//Envoie de l'heure en ASCII
   				//Activer l'envoie de la mailbox "heure"
   	   			*pCAN_TRS2 |= 0x0010;				
    	}
    }
    
    if((can_rx_flag&0x0002)==0x0002){ //Demande de la position	
    	//Envoyer la position via la l'id 21 si la trame est valide :   
    	if((nb_virgules>=2) && ((position_virgules[1]-position_virgules[0])>4)){ //si la trame est valide :     	
			//ENVOIE DE LA LONGITUDE ET DE LA LATITUDE EN BINAIRE AVEC UNE MAILBOX
   			//Eviter tout problmes dans le cas o il y a un point dans l'heure (qui influence le nombre de points dans la longitude/latitude)
   			if((position_points[0]<position_virgules[1]))
   				is_point_in_hour=1;
   			else
   				is_point_in_hour=0;
   			
   			if(		(position_points[0+is_point_in_hour]-position_virgules[1]-1)==4 //4 chiffres
   				&&  (position_virgules[2]-position_points[0+is_point_in_hour]-1)==4 //4 chiffres
   				&&	(position_virgules[3]-position_virgules[2]-1)==1 //1 char 
   				&&  (position_points[1+is_point_in_hour]-position_virgules[3]-1)==5 //5 chiffres
   				&&  (position_virgules[4]-position_points[1+is_point_in_hour]-1)==4 //4 chiffres
   				&&	(position_virgules[5]-position_virgules[4]-1)==1 //1 char 
   			)
   			{	
   			    
   			//Conversion des coordonnes GPS UART -> Trame binaire CAN : 
   			/*---------------------------------------------------------------
   			 *Trame UART :4907.8000,N,12303.6000,W 
   			 *Trame CAN   0x4907801230360
   			 *	                      ^
   			 *	                      4 bits de MSB : prcision des secondes
   			 *	                      LSB : 0->Nord / 1->Sud
   			 *	                      2eme LSB : 0->Ouest / 1->Est 				
   			 *---------------------------------------------------------------*/
   			buffer_long_lat[0]=(buffer_can[position_virgules[1]+1]<<4)+(buffer_can[position_virgules[1]+2]&0x0F);
   			buffer_long_lat[1]=(buffer_can[position_virgules[1]+3]<<4)+(buffer_can[position_virgules[1]+4]&0x0F);
   			buffer_long_lat[2]=(buffer_can[position_virgules[1]+6]<<4)+(buffer_can[position_virgules[1]+7]&0x0F);   			
   	   		buffer_long_lat[3]=(buffer_can[position_virgules[3]+1]<<4)+(buffer_can[position_virgules[3]+2]&0x0F);		

  		   	buffer_long_lat[4]=(buffer_can[position_virgules[3]+3]<<4)+(buffer_can[position_virgules[3]+4]&0x0F);		
  		   	buffer_long_lat[5]=(buffer_can[position_virgules[3]+5]<<4)+(buffer_can[position_virgules[3]+7]&0x0F);
  		   			
  		   	buffer_long_lat[6]=(buffer_can[position_virgules[3]+8]<<4);
  		   	if(buffer_can[position_virgules[2]+1]=='S')
  		   		buffer_long_lat[6]|= (1<<0);
  		   	else
  		   		buffer_long_lat[6]&= ~(1<<0);
  		   			
  		   	if(buffer_can[position_virgules[4]+1]=='E')
  		   		buffer_long_lat[6]|= (1<<1);
  		   	else
  		   		buffer_long_lat[6]&= ~(1<<1);
  		   	
  		   			
   	   		preparation_mailbox(buffer_long_lat,7,21); //Prparation de l'envoi des coordonnes GPS en "binaire"
 			
   	   		
   	   		//Activer l'envoie de la mailbox "position"
   	   		*pCAN_TRS2 |= 0x0020;
 			
   			}    	
    	
    	 	
    	}
     }     			
	}
}

//--------------------------------------------------------------------------//
// CAN TX IRQ (non ncessaire)												//
//--------------------------------------------------------------------------//
/**
 *@fn EX_INTERRUPT_HANDLER(Can_tx_ISR)
 *@brief Interruption de fin de transmission d'une trame CAN via le bus CAN
 *@details
 * Cette interruption se lance  la fin de l'envoie d'une trame CAN.
 * Cela permet d'diter un flag de fin de transmission de trame via bus CAN.
 *//*
EX_INTERRUPT_HANDLER(Can_tx_ISR){
    //Nettoyer flag d'envoie l'envoie de l'heure et de la position
    *pCAN_MBTIF2 =0x0030;    	   
}*/

//--------------------------------------------------------------------------//
// Initialisation des interruptions											//
//--------------------------------------------------------------------------//
/**
 *@fn void Init_Interrupts(void)
 *@brief Fonction d'initialisation des interruptions CAN et UART0-DMA9 ainsi que leurs priorits associes.
 *@details
 * Activation des interruptions : UART0-TX(DMA9), UART0-RX (GPS), CAN-RX (Demande), CAN-TX (fin d'envoie d'une trame CAN).
 * Ordre de priorit (du plus prioritaire au moins prioritaire) : UART0-TX > UART0-RX > CAN-TX > CAN-RX
 * Grce  cette ordre de priorit, il n'est pas possible d'envoyer une trame non valide via le bus CAN (sans compter la protection via le flag buffer_can_ready).
 */
 void Init_Interrupts(void)
{
	*pSIC_IMASK = 0x19800; // Activer les interruptions UART0 tx(dma)&rx et ceux du can
	
	*pCAN_MBIM2 = 0x0030; // Autoriser l'interuption de les mailbox 20 et 21 du CAN TX
	
	*pCAN_MBIM1 = 0x0003; // Autoriser l'interuption de la mailbox 0 et 1 du CAN RX.
	
	*pUART0_IER =0x03; // Autoriser interruptions UART0 RX et TX

	/*
	Les interruptions UART sont prioritaires car le buffer GPS 
	ne doit pas tre lu par le CAN pendant que l'on prpare celui-ci
	 */
		// assign core IDs to interrupts
	*pSIC_IAR0 = 0xffffffff;
	*pSIC_IAR1 = 0x3ff12fff; //UART0 rx ISR -> ID2, UART0 tx(dma) ISR -> ID1, CAN RX->ID0
	*pSIC_IAR2 = 0xffffffff; //Pas de CAN TX (si on veut l'utiliser : 0xfffffff0)				
	*pSIC_IAR3 = 0xffffffff;

	// assign ISRs to interrupt vectors
	register_handler(ik_ivg9,Uart0_ISR);		// UART0 rx ISR -> IVG 9
	register_handler(ik_ivg8,Uart0_tx_DMA_ISR);		// UART0 tx ISR -> IVG 8
	
	// Interruptions du CAN
	register_handler(ik_ivg10,Can_rx_ISR);		// CAN rx ISR -> IVG 10
	//register_handler(ik_ivg7,Can_tx_ISR);		// CAN tx ISR -> IVG 7
}
