/**
 *@file CAN.c
 *
 *@brief Fichier contenant les fonctions lies au CAN
 *@details
 * Vous trouverez ici les fonctions init_CAN() et preparation_mailbox().
 *@warning La fonction preparation_mailbox() doit tre appele aprs init_CAN().
 *@author Bayoumi Chamaa et Comte-Gaz Quentin
 *@date Novembre 2013
 *@version 1.0
 */
#include <cdefBF537.h>
#include <ccblkfn.h>
#include <sys/exception.h>

unsigned char buffer_can[100]; /*!< @brief Buffer dit par l'UART uniquement quand la trame est valide et est utilis par le CAN */
int buffer_can_ready; /*!< 
					   *@brief Protection vitant que l'UART dite le buffer_can[100] quand on remplie les mailboxes CAN 
					   *@details En thorie, cette variable est inutile car on a jou sur la priorit des interruptions pour garantir que le buffer sera valide lors d'une interruption CAN.
					   */

/**
 *@fn void init_CAN(void)
 *@brief Initialisation du bus CAN ainsi que des mailboxes  utiliser.
 *@details
 * Baudrate : 500 000 KHz (SJW = 1, TSEG2 = 5, TSEG1 = 12, BRP = 9)
 * Mailbox de transmission : 20 (heure, associe  l'ID 20) et 21 (position, associe  l'id 21)
 * Mailbox de reception : 0 (heure, associe  l'ID 20) et 1 (position, associe  l'ID 21)
 *@warning Cette fonction doit tre appele avant d'utiliser les autres fonctions lies au CAN.
 */
void init_CAN(void){
	short msgID;
	volatile char mbID;
	volatile char mbID_rx;
	
	buffer_can_ready=0; // Initialiser l'autorisation d'utiliser le buffer CAN
	
	/* On active le port J pour le bus CAN 
	 * sans toucher aux autres ports */
	*pPORT_MUX |= 0x02 ;  
	*pPORT_MUX &= ~(0x04) ;
	
	/* ===================================================
	 * Calcul du baudrate (SJW/TSEG2/TSEG1/BRP) :
	 * ===================================================
	 * Posons : SJW = 1, TSEG2 = 5, TSEG1 = 12
	 * On a bien SJW <= TSEG2 <= TSEG1
	 * ===================================================
	 * 500kHz CAN Clock (tBIT = 2us)
	 * ===================================================
	 * tBIT = TQ * (1 + (TSEG1 + 1) + (TSEG2 + 1))
	 * => 2e-6 = TQ * (1 + 13 + 6)
	 * => TQ = 1e-7
	 *
	 * TQ = (BRP+1) / SCLK
	 * => 1e-7 = (BRP+1) / 100e6
	 * => (BRP+1) = 10
	 * => BRP = 9 = CAN_CLOCK
	 * ==================================================*/
	*pCAN_TIMING = 0x015C; /* Posons SJW = 1, TSEG2 = 5, TSEG1 = 12 */
	*pCAN_CLOCK  = 9; /* On obtient alors BRP = 9 */
		
	msgID = 0x20;
	mbID_rx = 0;
	mbID  = 20;
	
	int j;
	for(j=20;j<22;j++){ 
		/* ID1, mask disabled, remote frame disable, 11 bit identifier */	
		/* Configurer la mailbox 20 et 21 (TX) */
		*(pCAN_MB_ID1(mbID)) = msgID << 2;
		*(pCAN_MB_LENGTH(mbID)) = 8;			/* DLC = 8 byte	
							*/
		/* Configurer la mailbox 0 et 1 (RX) */
		*(pCAN_MB_ID1(mbID_rx)) = msgID << 2;	
		
		mbID++;
		msgID++;
		mbID_rx++;
	}
	
	/* Set Mailbox Direction												        */
	/* *pCAN_MD1 et *pCAN_MD2 ... Pas besoin car dj fait automatiquement au reset */

	/* Enable Mailboxes															    */
	*pCAN_MC2 = 0x0030; 	/* Enables Mailbox 20-21 (TX)						    */
	*pCAN_MC1 = 0x0003;		/* Enables Mailbox 0-1 (RX)							    */
	
	/* Exit CAN Configuration Mode (Clear CCR)								 	    */
	*pCAN_CONTROL &= ~CCR;

	/* Wait for CAN Configuration Acknowledge (CCA) to Clear					    */
	while(*pCAN_STATUS & CCA);
}

/**
 *@fn void preparation_mailbox(unsigned char* data,int taille_data,int mbID)
 *@brief Prparation des mailbox d'envoie (uniquement l'heure et la position --> mailboxes 20 et 21)
 *@details
 * Cette fonction permet de remplir les registres contenant les 8 octets de donnes que l'on veut transmettre via bus CAN.
 *@warning Ne pas oublier d'activer la transmission CAN aprs avoir utilis cette fonction (pCAN_TRS2)
 *@param *data Pointeur vers le tableau de charactres  envoyer
 *@param taille_data nombre d'octets  envoyer (si superieur  8, alors on envoie 8 octets)
 *@param mbID Mailbox  remplir
 */
/* Prparation des mailbox d'envoie (uniquement l'heure et la position --> mailboxes 20 et 21) */
void preparation_mailbox(unsigned char* data,int taille_data,int mbID){
	int i;
	
	// Si on envoie moins de 8 octets, alors en remplie la MAILBOX d'elements inutiles (que l'on ne va pas envoyer)
	if(taille_data<8){ 
		for(i=taille_data;i<8;i++)
			data[i]=0;
		*(pCAN_MB_LENGTH(mbID)) = taille_data;	
	}
	else // Pour eviter les overflows
		*(pCAN_MB_LENGTH(mbID)) = 8;
	
	switch(mbID){
	    	
	// Remplit les donnes de trames dans les mailbox et activation envoie
	  case 20:
		*pCAN_MB20_DATA0=(data[6]<<8)+data[7];
		*pCAN_MB20_DATA1=(data[4]<<8)+data[5];	
		*pCAN_MB20_DATA2=(data[2]<<8)+data[3];		
		*pCAN_MB20_DATA3=(data[0]<<8)+data[1];
		//*pCAN_TRS2 |= 0x0010;		
		break;
		
	  case 21:
		*pCAN_MB21_DATA0=(data[6]<<8)+data[7];
		*pCAN_MB21_DATA1=(data[4]<<8)+data[5];	
		*pCAN_MB21_DATA2=(data[2]<<8)+data[3];		
		*pCAN_MB21_DATA3=(data[0]<<8)+data[1];
		//*pCAN_TRS2 |= 0x0020;
		break;
		
	  default:
	  	break;
	}
}
